#!/bin/sh

mode="$1"
relpath="$2"
destdir="$3"
time="$4"

verbosity=8
dryrun=""
#dryrun="--dry-run"
basedir="/var/duplicity/$(hostname -f)"
tmpdir="$basedir/tmp"
cachedir="$basedir/cache"
duplicity="/usr/bin/duplicity"
target="rclone://duplicity:/minmaxiste-bk-$(hostname -s)/duplicity"
email=root
file_list=/etc/duplicity.conf
lock="$basedir/.lock"

# With a bucket protection policy in place for 90 days, we keep a chain for
# ~60 days to ensure the initial full backup is protected. We keep 2 sets
# to ensure we don't attempt deleting before the protection period has
# expired to avoid backend errors.
full_retention=3
backup_chain_days=60
# Higher volume size to reduce the number of operations (cost) on the backend
multipart_chunk_size=250
volsize=1000

HOME=/root
PATH=/sbin:/usr/sbin:/bin:/usr/bin:/usr/local/sbin:/usr/local/bin

die()
{
	echo "$(basename $0): error: $1" >&2
	exit 1
}

[ "`whoami`" = "root" ] || die "must be run as root"

[ -x /usr/bin/bc ] || die "bc command not found"
[ -r /etc/duplicity_secrets ] \
	|| die "cannot read secrets file /etc/duplicity_secrets"
[ -r /etc/duplicity.conf ] || die "cannot read conf file /etc/duplicity.conf"

. /etc/duplicity_secrets

export PASSPHRASE
export RCLONE_CONFIG_DUPLICITY_TYPE="google cloud storage"
export RCLONE_CONFIG_DUPLICITY_SERVICE_ACCOUNT_FILE="/etc/duplicity_creds"
export RCLONE_CONFIG_DUPLICITY_LOCATION="us"
export RCLONE_CONFIG_DUPLICITY_STORAGE_CLASS="COLDLINE"
export RCLONE_CONFIG_DUPLICITY_BUCKET_POLICY_ONLY="true"

if [ ! -d "$cachedir" ]; then
	mkdir -p $cachedir
	chmod 700 $cachedir
fi

if [ ! -d "$tmpdir" ]; then
	mkdir -p $tmpdir
	chmod 700 $tmpdir
fi

touch "$lock" || die "cannot write lock file"

process() {
	if [ "$mode" = "full" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			$dryrun \
			--include-filelist $file_list \
			--volsize $volsize \
			--s3-multipart-chunk-size $multipart_chunk_size \
			$mode / $target \
			2>&1 | logger -t duplicity -p daemon.notice
	elif [ "$mode" = "run" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			--full-if-older-than "${backup_chain_days}D" \
			$dryrun \
			--include-filelist $file_list \
			--volsize $volsize \
			--s3-multipart-chunk-size $multipart_chunk_size \
			incr / $target \
			2>&1 | logger -t duplicity -p daemon.notice
	elif [ "$mode" = "list-current-files" -o "$mode" = "collection-status" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity 1 \
			$mode $target
	elif [ "$mode" = "verify" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			--include-filelist $file_list \
			$mode $target / \
			2>&1 | mail -s "Duplicity verify (to drive) for `hostname`" $email
	elif [ "$mode" = "verify-stdout" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			--include-filelist $file_list \
			verify $target /
	elif [ "$mode" = "cleanup" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			--force \
			$mode $target \
			2>&1 | logger -t duplicity -p daemon.notice
	elif [ "$mode" = "remove-old" ]; then
		$duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			remove-all-but-n-full $full_retention --force $target \
			2>&1 | logger -t duplicity -p daemon.notice
	elif [ "$mode" = "restore" ]; then
		if [ -z "$relpath" -o -z "$destdir" -o -z "$time" ]; then
			echo "Usage: $0 restore <relpath> <destdir> <time>"
			echo ""
			echo "    Example:"
			echo ""
			echo "    \$ $0 restore home/plalonde/.ckrts.json.gpg \\"
			echo "        /var/restore/.ckrts.json.gpg 2017-10-16"
			return 1
		fi
		# 2002-01-25T07:00:00, or YYYY-MM-DD

		echo $duplicity \
			--archive-dir $cachedir \
			--tempdir=$tmpdir \
			--verbosity $verbosity \
			$mode \
			--file-to-restore $relpath \
			--time $time \
			$target $destdir
	else
		echo "Usage: $0 <run|full|list|collection-status|verify|clean|remove-old|restore>"
		return 1
	fi

	return 0
}

(
	flock -n 3 || exit 2
	process
) 3>"$lock"

if [ $? -eq 2 ]; then
	die "flock failed; already running?"
fi

exit $?
