class pantin::apache_http_server {
	$fqdn = $facts['networking']['fqdn']

	define www(
		$ensure = 'present',
		$server_name = $facts['networking']['fqdn'],
		Array[String] $aliases = [],
		$docroot = '/var/www/html',
	) {
		$fqdn = $facts['networking']['fqdn']
		if $ensure == 'present' {
			file { "/etc/apache2/sites-available/${name}.conf":
				ensure => $ensure,
				mode => '0644',
				content => template("pantin/etc/apache2/sites-available/${name}.conf.erb"),
			} ->
			exec { "site-${name}-enable":
				command => "a2ensite $name",
				creates => "/etc/apache2/sites-enabled/${name}.conf",
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		} else {
			exec { "site-${name}-disable":
				command => "a2dissite $name",
				onlyif => "test -f /etc/apache2/sites-enabled/${name}.conf",
				notify => Service['apache2'],
				require => Package['apache2'],
			} ->
			file { "/etc/apache2/sites-available/${name}.conf":
				ensure => $ensure,
			}
		}

		file { "/var/log/apache2/${name}":
			mode => '0755',
			owner => 'root',
			group => 'adm',
			ensure => directory,
			before => Service['apache2'],
		}
	}

	define mod($ensure = 'present', $has_conf = false) {
		if $ensure == 'present' {
			exec { "mod-${name}-enable":
				command => "a2enmod $name",
				creates => "/etc/apache2/mods-enabled/${name}.load",
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		} else {
			exec { "mod-${name}-disable":
				command => "a2dismod $name",
				refreshonly => true,
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		}

		if $has_conf {
			file { "/etc/apache2/modules-available/${name}.conf":
				ensure => $ensure,
				mode => '0644',
				notify => Service['apache2'],
				content => template("pantin/etc/apache2/mods-available/${name}.conf.erb"),
			}
		}
	}

	package { ['apache2', 'javascript-common']:
		ensure => present,
	}

	exec { 'conf-javascript-enable':
		command => 'a2enconf javascript-common',
		creates => '/etc/apache2/conf-enabled/javascript-common.conf',
		notify => Service['apache2'],
		require => Package['apache2', 'javascript-common'],
	}

	file { '/etc/apache2/ports.conf':
		mode => '0644',
		content => "Listen 80\nListen 442\n",
	} ->
	file { '/var/www/html/index.html':
		ensure => absent,
	} ->
	service { 'apache2':
		ensure => 'running',
		enable => true,
		subscribe => [
			File["/etc/ssl/certs/$fqdn.pem"],
			File["/etc/ssl/private/$fqdn.pem"],
		],
		require => Package['apache2'],
	}

	pantin::apache_http_server::www { '000-default':
		ensure => absent,
	}
	pantin::apache_http_server::www { 'base':
		ensure => present,
	}

	pantin::apache_http_server::mod { [
		'ssl',
		'dav',
		'dav_fs',
		'ext_filter',
		'proxy',
		'proxy_http',
		'remoteip',
		'rewrite',
		'headers' ]:
		ensure => present,
	}

	file { '/var/www/html':
		mode => '0755',
		ensure => directory,
	}
	file { '/var/www/html/.well-known':
		mode => '0755',
		ensure => directory,
	}
	file { '/var/www/html/__special':
		ensure => absent,
	}
	file { '/var/www/html/md.css':
		mode => '0644',
		content => file('pantin/var/www/html/md.css'),
	}
	file { '/var/www/html/robots.txt':
		mode => '0644',
		content => file('pantin/var/www/html/robots.txt'),
	}
	file { '/var/www/md-to-html-template.html':
		mode => '0644',
		content => file('pantin/var/www/md-to-html-template.html'),
	}
}
