class pantin::smtpd(Boolean $is_relay = false, Sensitive $submission_secret = '') {
	$fqdn = $facts['networking']['fqdn']
	$hostname = $facts['networking']['hostname']
	$domain = $facts['networking']['domain']
	$os = $facts['os']['family']

	$aliases_path = $os ? {
		'OpenBSD' => '/etc/mail/aliases',
		default => '/etc/aliases',
	}

	if $is_relay {
		$smtpd_svc = 'postfix'
		package { ['postfix',
			'postfix-pcre',
			'opendkim',
			'opendkim-tools']:
			ensure => present,
		} ->
		file { '/etc/postfix/main.cf':
			mode => '0644',
			content => template('pantin/etc/postfix/main.cf.erb'),
		} ->
		file { '/etc/postfix/master.cf':
			mode => '0644',
			content => template('pantin/etc/postfix/master.cf.erb'),
		} ->
		file { '/etc/postfix/virtual':
			mode => '0644',
			content => "/^root@.*$/\troot\n",
		} ->
		file { '/etc/postfix/generic':
			ensure => absent,
		} ->
		file { '/etc/postfix/canonical':
			mode => '0644',
			content => template('pantin/etc/postfix/canonical.erb'),
		} ->
		file { '/etc/postfix/controlled_envelope_senders':
			mode => '0644',
			content => "@minmaxiste.com\tminmaxiste@minmaxiste.com\n",
		} ->
		file { '/etc/ssl/private/smtp-relay-key.pem':
			mode => '0600',
			content => lookup('pantin::letsencrypt::key'),
		} ->
		file { '/etc/ssl/certs/smtp-relay-crt.pem':
			mode => '0644',
			content => file('pantin/etc/ssl/certs/$hostname-crt.pem'),
		}

		user { 'postfix':
			groups => [ 'sasl', 'opendkim' ],
			require => Package['opendkim'],
		}

		exec { 'postfix-sasl-passwd':
			command => 'puppet lookup pantin::smtpd::submission_secret | tail -1 | cut -d: -f 3 | saslpasswd2 -p -f /var/spool/postfix/etc/sasldb2 -c -u minmaxiste.com pantin',
			creates => '/var/spool/postfix/etc/sasldb2',
		}

		exec { 'postfix-refresh-controlled_envelope_senders':
			command => 'postmap /etc/postfix/controlled_envelope_senders',
			creates => '/etc/postfix/controlled_envelope_senders.db',
			refreshonly => true,
			subscribe => File['/etc/postfix/controlled_envelope_senders'],
		}

		service { 'postfix':
			ensure => 'running',
			enable => true,
			subscribe => [
				File['/etc/ssl/private/smtp-relay-key.pem'],
				File['/etc/ssl/certs/smtp-relay-crt.pem'],
				File['/etc/postfix/main.cf'],
				File['/etc/postfix/master.cf'],
				File['/etc/postfix/virtual'],
				File['/etc/postfix/generic'],
				File['/etc/postfix/canonical'],
				File['/etc/postfix/controlled_envelope_senders'],
			],
			require => User['postfix'],
		}

		file { '/var/spool/postfix/opendkim':
			owner => 'opendkim',
			group => 'postfix',
			mode => '0755',
			ensure => 'directory',
		}
		file { '/etc/opendkim':
			mode => '0755',
			ensure => 'directory',
		}
		file { '/etc/opendkim.conf':
			mode => '0644',
			content => template('pantin/etc/opendkim.conf.erb'),
		}
		file { '/etc/opendkim/default.private':
			owner => 'opendkim',
			group => 'opendkim',
			mode => '0400',
			content => lookup('pantin::smtpd::dkim_key'),
		}
		service { 'opendkim':
			ensure => 'running',
			enable => true,
			subscribe => [
				File['/etc/opendkim/default.private'],
				File['/etc/opendkim.conf'],
			],
			require => Package['opendkim'],
		}
	} else {
		if $os != 'OpenBSD' {
			package { ['opensmtpd', 'mailutils']:
				ensure => present,
			}
		}

		$conf_path = $os ? {
			'OpenBSD' => '/etc/mail/smtpd.conf',
			default => '/etc/smtpd.conf',
		}
		$smtpd_group = $os ? {
			'OpenBSD' => '_smtpd',
			default => 'opensmtpd',
		}
		$smtpd_svc = $os ? {
			'OpenBSD' => 'smtpd',
			default => 'opensmtpd',
		}
		$submission_secret_path = $os ? {
			'OpenBSD' => '/etc/mail/submission_secret',
			default => '/etc/submission_secret',
		}
		$relayauth_path = $os ? {
			'OpenBSD' => '/etc/mail/relayauth',
			default => '/etc/relayauth',
		}
		file { $submission_secret_path:
			mode => '0640',
			group => $smtpd_group,
			content => template('pantin/etc/mail/submission_secret.erb'),
			notify => Service["$smtpd_svc"],
		} ->
		file { $conf_path:
			mode => '0644',
			content => $is_relay ? {
				true => template('pantin/etc/mail/relay-smtpd.conf.erb'),
				default => template('pantin/etc/mail/smtpd.conf.erb'),
			},
			notify => Service["$smtpd_svc"],
		} ->
		service { $smtpd_svc:
			ensure => 'running',
			enable => true,
		}
	}

	if $os == 'OpenBSD' {
		file { '/etc/mail/aliases':
			mode => '0644',
			content => file('pantin/etc/mail/aliases'),
			notify => Exec['refresh-aliases'],
		}
	} else {
		mailalias { 'root':
			ensure => present,
			recipient => 'plalonde@minmaxiste.com',
			target => $aliases_path,
			notify => Exec['refresh-aliases'],
		}
	}

	exec { 'refresh-aliases':
		command => 'newaliases',
		refreshonly => true,
		notify => Service["$smtpd_svc"],
	}
}
