class pantin::apache_http_server(Boolean $setup_default_site = true) {
	$fqdn = $facts['networking']['fqdn']

	define www(
		$ensure = 'present',
		$template = 'base',
		Array[String] $aliases = [],
		$docroot = '/var/www/html',
		$docroot_gid = 'root',
		$docroot_mode = '0755',
		$docroot_options = [ 'MultiViews', 'FollowSymlinks' ],
	) {
		$fqdn = $facts['networking']['fqdn']
		if $ensure == 'present' {
			file { "/etc/apache2/sites-available/${name}.conf":
				ensure => $ensure,
				mode => '0644',
				content => template("pantin/etc/apache2/sites-available/${template}.conf.erb"),
			} ->
			exec { "site-${name}-enable":
				command => "a2ensite $name",
				creates => "/etc/apache2/sites-enabled/${name}.conf",
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		} else {
			exec { "site-${name}-disable":
				command => "a2dissite $name",
				onlyif => "test -L /etc/apache2/sites-enabled/${name}.conf",
				notify => Service['apache2'],
				require => Package['apache2'],
			} ->
			file { "/etc/apache2/sites-available/${name}.conf":
				ensure => $ensure,
			}
		}

		file { "$docroot":
			mode => $docroot_mode,
			owner => 'root',
			group => $docroot_gid,
			ensure => $ensure ? {
				absent => absent,
				default => directory,
			},
			before => Service['apache2'],
		}

		file { "/var/log/apache2/${name}":
			mode => '0755',
			owner => 'root',
			group => 'adm',
			ensure => $ensure ? {
				absent => absent,
				default => directory,
			},
			before => Service['apache2'],
		}

		file { "$docroot/.well-known":
			mode => '0755',
			ensure => $ensure ? {
				absent => absent,
				default => directory,
			},
		}
		file { "$docroot/md.css":
			ensure => $ensure,
			mode => '0644',
			content => file('pantin/var/www/md.css'),
		}
	}

	define mod($ensure = 'present', $has_conf = false) {
		if $ensure == 'present' {
			exec { "mod-${name}-enable":
				command => "a2enmod $name",
				creates => "/etc/apache2/mods-enabled/${name}.load",
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		} else {
			exec { "mod-${name}-disable":
				command => "a2dismod $name",
				refreshonly => true,
				notify => Service['apache2'],
				require => Package['apache2'],
			}
		}

		if $has_conf {
			file { "/etc/apache2/modules-available/${name}.conf":
				ensure => $ensure,
				mode => '0644',
				notify => Service['apache2'],
				content => template("pantin/etc/apache2/mods-available/${name}.conf.erb"),
			}
		}
	}

	package { ['apache2', 'javascript-common']:
		ensure => present,
	}

	file { '/var/www/md-to-html-template.html':
		mode => '0644',
		content => file('pantin/var/www/md-to-html-template.html'),
		require => Package['apache2'],
	}

	exec { 'conf-javascript-enable':
		command => 'a2enconf javascript-common',
		creates => '/etc/apache2/conf-enabled/javascript-common.conf',
		notify => Service['apache2'],
		require => Package['apache2', 'javascript-common'],
	}

	file { '/etc/apache2/sites-available/default-ssl.conf':
		ensure => absent,
	}

	file { '/etc/apache2/ports.conf':
		mode => '0644',
		content => "Listen 80\nListen 442\n",
		require => Package['apache2'],
	} ->
	service { 'apache2':
		ensure => 'running',
		enable => true,
		subscribe => [
			File["/etc/ssl/certs/$fqdn.pem"],
			File["/etc/ssl/private/$fqdn.pem"],
		],
		require => Package['apache2'],
	}

	pantin::apache_http_server::www { '000-default':
		ensure => absent,
	}

	if $setup_default_site {
		pantin::apache_http_server::www { "$fqdn":
			ensure => present,
		}
	}

	pantin::apache_http_server::mod { [
		'ssl',
		'dav',
		'dav_fs',
		'ext_filter',
		'proxy',
		'proxy_http',
		'remoteip',
		'rewrite',
		'headers' ]:
		ensure => present,
	}
}
